<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon; // Ensure Carbon is imported

class UserSignalSubscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'signal_id',
        'start_date',
        'end_date',
        'paid_price',
        'status', // 'active', 'expired', 'canceled'
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'paid_price' => 'float',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function signal()
    {
        return $this->belongsTo(Signal::class);
    }

    /**
     * Accessor to get remaining days as a whole number.
     */
    public function getRemainingDaysAttribute(): int
    {
        if ($this->status !== 'active' || $this->end_date->isPast()) {
            return 0;
        }
        // Use diffInDays for whole days, ensuring it's an integer type
        return (int) now()->diffInDays($this->end_date, true); // `true` for absolute difference
    }

    /**
     * Accessor to get progress percentage as a whole number (rounded).
     */
    public function getProgressPercentageAttribute(): int
    {
        if ($this->start_date === null || $this->end_date === null) {
            return 0; // Or handle as appropriate if dates are not set
        }

        $now = now();
        if ($this->status !== 'active' || $this->end_date->isPast() || $now->greaterThanOrEqualTo($this->end_date)) {
            return 100; // If expired or past end date, progress is 100%
        }
        if ($now->lessThanOrEqualTo($this->start_date)) {
            return 0; // If not started yet, progress is 0%
        }

        $totalDurationInSeconds = $this->start_date->diffInSeconds($this->end_date);
        $elapsedDurationInSeconds = $this->start_date->diffInSeconds($now);

        if ($totalDurationInSeconds <= 0) { // Avoid division by zero or negative duration
            return 100;
        }

        return (int) round(($elapsedDurationInSeconds / $totalDurationInSeconds) * 100);
    }
}
