<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Carbon\Carbon;

class UserFixedInvestment extends Model
{
    use HasFactory, SoftDeletes;

    // ... (Constants for STATUS_PENDING, STATUS_ACTIVE, etc.)
    const STATUS_PENDING = 'pending';
    const STATUS_ACTIVE = 'active';
    const STATUS_MATURED = 'matured';
    const STATUS_DISBURSED = 'disbursed';
    const STATUS_CANCELED = 'canceled';

    protected $fillable = [
        'user_id',
        'coin_id',
        'inv_id',
        'investment_plan_id',
        'invested_amount',
        'current_pnl',
        'pnl_amount',
        'start_date',
        'end_date',
        'status',
        'user_chosen_duration_value',
        'user_chosen_duration_unit',
        'disbursed_amount',
        'disbursed_by_id',
        'disbursed_at',
    ];

    protected $casts = [
        'invested_amount' => 'float',
        'current_pnl' => 'float',
        'pnl_amount' => 'float',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'disbursed_at' => 'datetime',
        'user_chosen_duration_value' => 'integer',
    ];

    protected $appends = ['days_left', 'days_gone', 'percent_completed'];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function investmentPlan()
    {
        return $this->belongsTo(InvestmentPlan::class);
    }

    public function coin()
    {
        return $this->belongsTo(Coin::class, 'coin_id');
    }

    public function disbursedBy()
    {
        return $this->belongsTo(User::class, 'disbursed_by_id');
    }

    // Accessors for display (similar to Investment for staking)
    // Note: These accessors now calculate based on actual start/end dates,
    // which are derived from user_chosen_duration_value/unit
    public function getDaysLeftAttribute()
    {
        if (is_null($this->end_date) || $this->end_date->isPast() || $this->status === 'disbursed' || $this->status === 'canceled') {
            return 0;
        }
        return (int) round(now()->diffInDays($this->end_date, false));
    }

    public function getDaysGoneAttribute()
    {
        if (is_null($this->start_date)) {
            return 0;
        }
        $now = now();
        if ($now->lessThanOrEqualTo($this->start_date)) {
            return 0;
        }
        if ($this->end_date && $now->greaterThanOrEqualTo($this->end_date)) {
            return (int) $this->start_date->diffInDays($this->end_date, false);
        }
        return (int) $this->start_date->diffInDays($now, false);
    }

    public function getPercentCompletedAttribute()
    {
        if (is_null($this->start_date) || is_null($this->end_date) || $this->start_date->equalTo($this->end_date) || $this->days_left == 0) {
            return 100; // Or 0 if not started, depends on context
        }
        $now = now();
        if ($now->greaterThanOrEqualTo($this->end_date)) {
            return 100;
        }
        if ($now->lessThanOrEqualTo($this->start_date)) {
            return 0;
        }

        $totalDurationInSeconds = $this->start_date->diffInSeconds($this->end_date);
        $elapsedDurationInSeconds = $this->start_date->diffInSeconds($now);

        if ($totalDurationInSeconds <= 0) return 100;

        return (int) round(min(100, max(0, ($elapsedDurationInSeconds / $totalDurationInSeconds) * 100)));
    }

    public function getCurrentPnlXAttribute()
    {
        $value = ($this->getPercentCompletedAttribute() / 100) * ($this->investmentPlan->daily_roi_percentage / 100) * $this->invested_amount;
        return $value;
    }
}
