<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;

class Stock extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'code',
        'value',
        'ask_value',
        'ask_shares',
        'bid_value',
        'bid_shares',
        'year_range_min',
        'year_range_max',
        'market',
        'tradingview_code',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'value' => 'decimal:2',
        'ask_value' => 'decimal:2',
        'bid_value' => 'decimal:2',
        'year_range_min' => 'decimal:2',
        'year_range_max' => 'decimal:2',
        'ask_shares' => 'integer',
        'bid_shares' => 'integer',
    ];

    protected $appends = ['claimed', 'unclaimed'];

    public function wallet()
    {
        return $this->hasOne(StockWallet::class)->where('user_id', Auth::id());
    }

    public function getClaimedAttribute()
    {
        // Access the wallet relationship and then get its 'claimed' attribute
        $wallet = $this->wallet;

        // Use the optional helper '?' to prevent errors if the wallet doesn't exist.
        return $wallet->claimed ?? 0;
    }

    public function getUnclaimedAttribute()
    {
        // Access the wallet relationship and then get its 'unclaimed' attribute
        $wallet = $this->wallet;
        
        // Use the optional helper '?' to prevent errors if the wallet doesn't exist.
        return $wallet->unclaimed ?? 0;
    }

    // public function geClaimedBalanceAttribute()
    // {
    //     return $this->hasOne(StockWallet::class)->where('user_id', Auth::id())->sum('claimed');
    // }

    // public function getUnclaimedBalanceAttribute()
    // {
    //     return $this->hasOne(StockWallet::class)->where('user_id', Auth::id())->sum('unclaimed');
    // }

    public function totalClaimedForCurrentUser()
    {
        return $this->hasOne(StockWallet::class)
            ->where('user_id', Auth::id())
            ->selectRaw('stock_id, SUM(claimed) as total_claimed')
            ->groupBy('stock_id');
    }

    /**
     * Get the total unclaimed amount of this stock for the authenticated user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function totalUnclaimedForCurrentUser()
    {
        return $this->hasOne(StockWallet::class)
            ->where('user_id', Auth::id())
            ->selectRaw('stock_id, SUM(unclaimed) as total_unclaimed')
            ->groupBy('stock_id');
    }
}
