<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Investment extends Model
{
    use HasFactory, SoftDeletes;


    const STATUS_PENDING = 1;
    const STATUS_ACTIVE = 2;
    const STATUS_MATURED = 3;
    const STATUS_DISBURSED = 4;
    const STATUS_CANCELED = 5;

    protected $fillable = [
        'user_id',
        'inv_id',
        'amount',
        'pnl_amount',
        'disbursed_amount',
        'pnl',
        'ended_at',
        'disbursed_by',
        'investment_plan_id',
        'duration',
        'roi',
        'disbursed_at',
        'status_id',
        // 'status',
    ];

    protected $dates = [
        'ended_at',
        'disbursed_at',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $casts = [
        'ended_at' => 'datetime',
        'disbursed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
        'amount' => 'float',
        'pnl_amount' => 'float',
        'disbursed_amount' => 'float',
        'roi' => 'float',
        'duration' => 'integer',
        'user_id' => 'integer',
        // 'status' => 'integer',
        'status_id' => 'integer',
        'investment_plan_id' => 'integer',
    ];

    protected static function booted()
    {
        static::retrieved(function ($investment) {
            // Ensure investment has an investment plan
            if (
                $investment->status_id == Investment::STATUS_ACTIVE &&
                $investment->created_at->addDays($investment->duration)->lte(now())
            ) {
                // Update status to matured
                $investment->status_id = Investment::STATUS_MATURED;

                // If ended_at is null, set it to created_at + duration
                if (is_null($investment->ended_at)) {
                    $investment->ended_at = $investment->created_at->addDays($investment->duration);
                }

                // Save the changes
                $investment->save();
            }
        });
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function disbursedBy()
    {
        return $this->belongsTo(User::class, 'disbursed_by');
    }

    public function status()
    {
        return $this->belongsTo(InvestmentStatus::class);
    }

    public function statusRel() // Relationship method name
    {
        return $this->belongsTo(InvestmentStatus::class, 'status_id');
    }

    public function coin()
    {
        return $this->belongsTo(Coin::class, 'investment_plan_id');
    }

    public function getDaysLeftAttribute()
    {
        if (!$this->ended_at) {
            $daysLeft = $this->duration - abs(now()->diffInDays($this->created_at));
            // return round(now()->diffInDays($this->created_at));
            if ($daysLeft < 0) {
                return 0;
            }
            return round($daysLeft);
        }

        return 0;
    }

    public function getDaysGoneAttribute()
    {
        if (!$this->ended_at) {
            $daysGone = abs(now()->diffInDays($this->created_at));
            return round($daysGone);
        }
        return $this->duration;
    }

    public function getPercentCompletedAttribute()
    {
        if (!$this->ended_at) {
            $daysLeft = $this->duration - now()->diffInDays($this->created_at, true);
            $percentCompleted = ($this->duration - $daysLeft) / $this->duration * 100;
            // $percentCompleted = ($this->duration - $daysLeft) / $this->duration * 100;
            if ($percentCompleted>100) {
                // to catch some cases where calaculation resulted in greater than 100
                // !cause why fix an issue when you cann just patch it :)
                return 100;
            }
            return round($percentCompleted);
        }

        return 100;
    }

    // Your existing scopes...
    public function scopeActive($query)
    {
        return $query->where('status_id', self::STATUS_ACTIVE);
    }

    public function scopeMatured($query)
    {
        return $query->where('status_id', self::STATUS_MATURED);
    }

    public function scopeDisbursed($query)
    {
        return $query->where('status_id', self::STATUS_DISBURSED);
    }
}
