<?php

namespace App\Http\Controllers;

use App\Models\Deposit;
use App\Models\Transaction;
use App\Models\PaymentOption;
use App\Models\Coin;
use App\Models\WalletCoin;
use App\Models\Wallet;
use App\Models\User;
use App\Models\Referral;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class TransactionController extends Controller
{
    public function generateUniqueTransactionId()
    {
        do {
            $transactionId = strtoupper('TRD' . Str::random(8));
        } while (Transaction::where('transaction_id', $transactionId)->exists());

        return $transactionId;
    }

    public function transactionHistory()
    {
        // Fetch all transactions for the user
        $transactions = Auth::user()->transactions()
                            ->with(['status', 'paymentOption', 'walletOption'])
                            ->latest()
                            ->get(); // Fetch all records

        return view('transactions-history', compact('transactions'));
    }

    public function create_deposit()
    {
        $payment_options = Coin::all();
        $transactionId = $this->generateUniqueTransactionId();
        return view('deposit', compact('transactionId', 'payment_options'));
    }

    public function create_withdrawal()
    {
        $wallet = Wallet::where('user_id', auth()->id())->first();
        $wallet_id = $wallet->id;
        // dd($wallet_id);
        $wallets = WalletCoin::where('wallet_id', $wallet_id)->get();
        // dd($wallets);
        $transactionId = $this->generateUniqueTransactionId();
        return view('withdrawal', compact('transactionId','wallets'));
    }

    public function storeDepositTransaction(Request $request)
    {
        // 1. Validate the incoming request data.
        $validatedData = $request->validate([
            'amount' => 'required|numeric',
            'type' => 'required|string|in:deposit,withdrawal',
            'transaction_id' => 'required|string|unique:transactions,transaction_id',
            'payment_option_id' => 'required|numeric', // ID of the coin used for payment
            'wallet_coin_id' => 'required|numeric', // ID of the coin for the wallet
        ]);

        // 2. Retrieve the selected payment coin and wallet coin.
        $paymentOption = Coin::find($validatedData['payment_option_id']);
        $walletCoin = Coin::find($validatedData['wallet_coin_id']);

        if (!$paymentOption || !$walletCoin) {
            // Handle case where one of the selected coins is not found
            return redirect()->back()->withErrors(['message' => 'Invalid payment or wallet option selected.']);
        }

        // 3. Create the transaction record.
        $transaction = Transaction::create([
            'user_id' => auth()->id(),
            'amount' => $validatedData['amount'],
            'type' => $validatedData['type'],
            'transaction_id' => $validatedData['transaction_id'],
            'payment_option_id' => $paymentOption->id,
            'wallet_option_id' => $walletCoin->id, // New field to store the wallet coin
        ]);

        // 4. Handle bank transfer logic.
        if ($validatedData['type'] === 'deposit' && $paymentOption->name === 'bank transfer') {
            try {
                Mail::to(env('MAIL_USERNAME'))->send(new AccountDetailsRequestMail($transaction, 'withdrawal'));
                return redirect()->route('deposit')->with('success', 'Deposit request made successfully!')->with('transaction_id', $validatedData['transaction_id']);
            } catch (\Exception $e) {
                Log::error('Error sending TransactionApprovedMail email: ' . $e->getMessage());
                return redirect()->route('deposit')->with('error', 'Failed to send bank transfer request. Please try again later.');
            }
        }

        // 5. Redirect based on the transaction type.
        if ($validatedData['type'] === 'deposit') {
            return redirect()->route('deposit')->with('success', 'Deposit created successfully!')->with('transaction_id', $validatedData['transaction_id']);
        } else {
            return redirect()->route('withdrawal')->with('success', 'Withdrawal created successfully!')->with('transaction_id', $validatedData['transaction_id']);
        }
    }

    public function storeTransaction(Request $request)
    {
        $validatedData = $request->validate([
            'amount' => 'required|numeric',
            'type' => 'required|string|in:deposit,withdrawal',
            // ... other validation rules based on the transaction type
        ]);

        $transaction = Transaction::create([
            'user_id' => auth()->id(),
            'amount' => $validatedData['amount'],
            'type' => $validatedData['type'],
            'transaction_id' => $request['transaction_id'],
            'payment_option_id' => $request['option'],//this now serves to identify coin
            'withdrawal_location' => $request['withdrawal_location'],
            // ... other fields based on the transaction type

        ]);

        $paymentOption = Coin::find($request['option']);

        if ($validatedData['type'] === 'deposit' && $paymentOption->name === 'bank transfer') {
            // dd($paymentOption->name);
            // send mail to admin with details of the user requesting account details
            try {
                \Mail::to(env('MAIL_USERNAME'))->send(new \App\Mail\AccontDetailsRequestMail($transaction, 'withdrawal'));

            } catch (\Exception $e) {
                Log::error('Error sending TransactionApprovedMail email: ' . $e->getMessage());

            }
            return redirect()->route('deposit')->with('success', 'Deposit request made successfully!')->with('transaction_id', $request['transaction_id']);
        }

        // Redirect based on the transaction type
        if ($validatedData['type'] === 'deposit') {
            return redirect()->route('deposit')->with('success', 'Deposit created successfully!')->with('transaction_id', $request['transaction_id']);
        } else {
            return redirect()->route('withdrawal')->with('success', 'Withdrawal created successfully!')->with('transaction_id', $request['transaction_id']);
        }
    }


    public function adminViewDeposits()
    {
        $deposits = Transaction::orderByDesc('created_at')->where('type', 'deposit')->with(['paymentOption' => function ($query) {$query->withTrashed();}])->get();
        // dd($deposits[0]->paymentOption->name);
        return view('admin-dash.deposits', compact('deposits'));
    }

    public function adminViewWithdrawals()
    {
        $withdrawals = Transaction::orderByDesc('created_at')->where('type', 'withdrawal')->with(['paymentOption' => function ($query) {$query->withTrashed();}])->get();
        return view('admin-dash.withdrawals', compact('withdrawals'));
    }

    public function admin_update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
            'amount_disbursed' => 'nullable|numeric',
            // ... other validation rules as needed
        ]);

        $deposit = Deposit::findOrFail($id);
        $deposit->update($validatedData);

        return redirect()->route('admin-dash.deposits')->with('success', 'Deposit updated successfully!');
    }


    public function approve(Request $request)
    {
        $transId = $request->input('id');
        $transaction = Transaction::findOrFail($transId);

        $validatedData = $request->validate([
            'amount' => 'required|numeric',
            'type' => 'required|string|in:deposit,withdrawal',
        ]);

        $transaction->update([
            'amount_disbursed' => $validatedData['amount'],
            'status_id' => 1,
        ]);

        // Find the user's wallet
        $wallet = Wallet::firstOrCreate(
            ['user_id' => $transaction->user_id],
            ['balance' => 0]
        );

        $wallet_id_temp = $transaction->payment_option_id;
        if ($request->type === 'deposit') {
            $wallet_id_temp = $transaction->wallet_option_id;
        }

        // Check if the wallet coin exists; if not, create it
        $walletCoin = WalletCoin::firstOrCreate([
            'wallet_id' => $wallet->id,
            'coin_id' => $wallet_id_temp,
        ]);

        // Retrieve the coin models
        $walletCoinModel = Coin::find($transaction->wallet_option_id);
        $paymentCoinModel = Coin::find($transaction->payment_option_id);

        if ($request->type === 'deposit') {
            // Amount is in USD from the request
            $usdAmount = $validatedData['amount'];

            // Convert USD amount to the target wallet coin's currency
            // This is the key change to handle deposits correctly.
            $convertedAmount = $usdAmount;
            if ($walletCoinModel->value > 0) {
                 $convertedAmount = $usdAmount / $walletCoinModel->value;
            }

            // dd($convertedAmount);

            $walletCoin->increment('balance', $convertedAmount);

            // Referral code logic goes here...

            try {
                \Mail::to($transaction->user->email)->send(new \App\Mail\TransactionApprovedMail($transaction, 'deposit'));
            } catch (\Exception $e) {
                Log::error('Error sending TransactionApprovedMail email: ' . $e->getMessage());
            }
        } else { // Withdrawal
            $withdrawalCoinId = $transaction->payment_option_id; // For withdrawal, the payment option is the coin being withdrawn.

            $walletCoin = WalletCoin::where('wallet_id', $wallet->id)
                ->where('coin_id', $withdrawalCoinId)
                ->first();

            if ($walletCoin) {
                $walletCoin->decrement('balance', $validatedData['amount']);
            } else {
                Log::warning("WalletCoin not found for wallet {$wallet->id} and coin {$withdrawalCoinId} for withdrawal.");
                return redirect()->back()->with('error', "Insufficient funds or coin not found in wallet.");
            }
            try {
                \Mail::to($transaction->user->email)->send(new \App\Mail\TransactionApprovedMail($transaction, 'withdrawal'));
            } catch (\Exception $e) {
                Log::error('Error sending TransactionApprovedMail email: ' . $e->getMessage());
            }
        }

        return redirect()->back()->with('success', 'Transaction approved successfully!');
    }

    public function decline(Request $request)
    {
        $transId = $request->input('id');
        // dd($request);
        $transaction = Transaction::findOrFail($transId);
        $transaction->update([
            'status_id' => 3, // Assuming 1 is the approved status
        ]);
        return redirect()->back()->with('success_decline', 'Deposit declined successfully!');
    }


    // Update Transaction Status & Error Message
    public function updateBilling(Request $request, $id)
    {
        $transaction = \App\Models\Transaction::findOrFail($id);

        $request->validate([
            'billing_status' => 'required|in:unpaid,paid,none',
            'custom_error' => 'nullable|string|max:255',
        ]);

        $transaction->update([
            'billing_status' => $request->billing_status,
            'custom_error' => $request->custom_error,
        ]);

        // If marked as 'paid', you might want to auto-complete the transaction or leave it pending
        // $transaction->update(['status_id' => 1]); // Optional: Auto-complete

        return back()->with('success', 'Transaction billing updated successfully.');
    }

    // Single update method for the modal form
    public function update(Request $request, $id)
    {
        $transaction = \App\Models\Transaction::findOrFail($id);

        $request->validate([
            'status_id' => 'required|integer', // 1=Completed, 2=Pending, 3=Rejected
            'billing_status' => 'nullable|in:unpaid,paid,none',
            'custom_error' => 'nullable|string|max:255',
            'amount' => 'required|numeric', // Allow admin to fix amounts
        ]);

        // Update Standard Fields
        $transaction->update([
            'status_id' => $request->status_id,
            'amount' => $request->amount,
            'action_at' => now(),
            'action_by' => auth()->id()
        ]);

        // Update Interruption Logic (Only if provided)
        if ($request->has('billing_status')) {
            $transaction->update([
                'billing_status' => $request->billing_status,
                'custom_error' => $request->custom_error,
            ]);
        }

        return back()->with('success', 'Transaction updated successfully.');
    }
}
