<?php

namespace App\Http\Controllers;

use App\Models\Stock;
use App\Models\StockTrade;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

// use App\Http\Controllers\Controller;

class TradeStockController extends Controller
{
    private function generateUniqueTransactionId()
    {
        do {
            $transactionId = strtoupper('TRD' . Str::random(8));
        } while (Transaction::where('transaction_id', $transactionId)->exists());

        return $transactionId;
    }
    /**
     * Display a paginated list of the user's trade history.
     */
    public function history()
    {
        $trades = Auth::user()->stockTrades()
                        ->with('stock') // Eager load the stock relationship
                        ->orderBy('executed_at', 'desc')
                        ->paginate(20);

        return view('stocktrade-history', compact('trades'));
    }

    /**
     * Stores a new stock trade.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storeTrade(Request $request)
    {
        $request->validate([
            'trade_action' => ['required', Rule::in(['buy', 'sell'])],
            'selected_stock_id' => 'required|exists:stocks,id',
            'account_type' => ['required', Rule::in(['gbx_balance', 'usd_balance'])],
            'order_price_type' => ['required', Rule::in(['market', 'limit', 'stop', 'trailing_stop_limit'])],
            'buy_amount' => 'nullable|numeric|min:0.01',
            'sell_shares' => 'nullable|integer|min:1',
            // Add validation rules for limit price, stop price, trailing stop details if needed
        ]);

        $user = Auth::user();
        $stock = Stock::findOrFail($request->selected_stock_id);
        $transactionAccount = $stock->market;
        $tradeAction = $request->trade_action;
        $accountType = $request->account_type;
        $orderPriceType = $request->order_price_type;

        if ($transactionAccount === 'gbx') {
            $accountType = 'gbx_balance';
        } else {
            $accountType = 'usd_balance';
        }


        // dd($transactionAccount);

        // Map the account type to the enum value
        // $transactionAccount = null;
        // if ($accountType === 'gbx_balance') {
        //     $transactionAccount = 'gbx';
        // } elseif ($accountType === 'usd_balance') {
        //     $transactionAccount = 'usd';
        // }

        DB::beginTransaction();

        try {
            $stockTrade = StockTrade::create([
                'user_id' => $user->id,
                'stock_id' => $stock->id,
                'trade_action' => $tradeAction,
                'account_type' => $accountType,
                'order_type' => $orderPriceType,
                'executed_at' => now(),
                'quantity' => 0, // Initialize quantity
                'price_per_share' => 0, // Initialize price_per_share
            ]);

            if ($tradeAction === 'buy') {
                $buyAmount = $request->buy_amount;
                $stockPrice = $stock->value;
                $sharesToBuy = floor($buyAmount / $stockPrice);
                $costOfShares = $sharesToBuy * $stockPrice;

                $user->wallet()->decrement($accountType, $costOfShares);
                $user->stockWallets()->firstOrCreate(
                    ['stock_id' => $stock->id],
                    ['claimed' => 0, 'unclaimed' => 0]
                )->increment('unclaimed', $sharesToBuy);

                // Create a Transaction record for the buy order
                Transaction::create([
                    'user_id' => $user->id,
                    'transaction_id' => $this->generateUniqueTransactionId(),
                    'amount' => $costOfShares,
                    'details' => "Bought {$sharesToBuy} shares of {$stock->name} ({$stock->symbol}) at " . number_format($stockPrice * $sharesToBuy, 2) . " " . strtoupper(explode('_', $accountType)[0]),
                    'type' => 'stock_buy',
                    'status_id' => 1,
                    'action_at' => now(),
                    'action_by' => $user->id,
                    'account' => $transactionAccount,
                    'stock_trade_id' => $stockTrade->id,
                ]);

                $stockTrade->update([
                    'quantity' => $sharesToBuy,
                    'price_per_share' => $stockPrice,
                ]);

                DB::commit();
                return back()->with('success_stock_trade', "Successfully bought {$sharesToBuy} shares of {$stock->name} ({$stock->symbol}) at " . number_format($stockPrice * $sharesToBuy, 2) . " " . strtoupper(explode('_', $accountType)[0]));
            } elseif ($tradeAction === 'sell') {
                $sellShares = $request->sell_shares;
                $userStockWallet = $user->stockWallets()->where('stock_id', $stock->id)->first();
                $stockPrice = $stock->value;
                $sellAmount = $sellShares * $stockPrice;

                $userStockWallet->decrement('claimed', $sellShares);
                $user->wallet()->increment($accountType, $sellAmount);
                $userStockWallet->increment('sold', $sellShares);

                // Create a Transaction record for the sell order
                Transaction::create([
                    'user_id' => $user->id,
                    'transaction_id' => $this->generateUniqueTransactionId(),
                    'amount' => $sellAmount,
                    'details' => "Sold {$sellShares} shares of {$stock->name} ({$stock->symbol}) at " . number_format($stockPrice, 2) . " " . strtoupper(explode('_', $accountType)[0]),
                    'type' => 'stock_sell',
                    'status_id' => 1,
                    'action_at' => now(),
                    'action_by' => $user->id,
                    'account' => $transactionAccount,
                    'stock_trade_id' => $stockTrade->id,
                ]);

                $stockTrade->update([
                    'quantity' => $sellShares,
                    'price_per_share' => $stockPrice,
                ]);

                $currencySymbol = strtoupper(explode('_', $accountType)[0]) === 'USD' ? '$' : '£';

                DB::commit();
                $message = "<div class='text-left space-y-2 max-w-[300px]'>
                                <p class='text-base text-gray-700 dark:text-gray-300'>{$stock->name}</p>
                                <p class='text-sm text-gray-500 dark:text-gray-400'>Share Symbol: {$stock->symbol}</p>
                                <p class='text-sm text-gray-500 dark:text-gray-400'>Share Price As of Today: " . $currencySymbol . number_format($stockPrice, 2) . "</p>
                                <p class='text-sm text-gray-500 dark:text-gray-400'>Number of Shares: {$sellShares}</p>
                                <p class='text-sm text-gray-500 dark:text-gray-400'>Market: ".strtoupper($stock->market)."</p>
                                <p class='font-semibold text-base text-green-600 dark:text-green-400'>Total Value of Shares Sold: " . $currencySymbol . number_format($sellAmount, 2) . "</p>
                                <p class='text-gray-500 dark:text-gray-400 text-sm'>Funds have been added to your " . strtoupper(explode('_', $accountType)[0]) . " balance.</p>
                            </div>";
                return redirect()->route('dashboard')->with('success_stock_trade', $message);
            }

            DB::rollBack();
            return back()->withErrors(['error' => 'Invalid trade action.']);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error processing trade: ' . $e->getMessage());
            return back()->withErrors(['error' => 'An error occurred while processing your trade. Please try again.']);
        }
    }

    public function getStockDetails($code)
    {
        $stock = Stock::where('code', $code)->first();

        if (!$stock) {
            return response()->json(['error' => 'Stock not found'], 404);
        }

        $user = Auth::user();
        $stockWallet = $user->stockWallets()->where('stock_id', $stock->id)->first();

        return response()->json([
            'id' => $stock->id,
            'code' => $stock->code,
            'name' => $stock->name,
            'symbol' => $stock->symbol,
            'market' => $stock->market,
            'value' => $stock->value,
            'ask_value' => $stock->value * 1.001, // Mock ask price (slightly higher)
            'bid_value' => $stock->value * 0.999, // Mock bid price (slightly lower)
            'year_range_max' => $stock->year_range_max ?? $stock->value * 1.2,
            'year_range_min' => $stock->year_range_min ?? $stock->value * 0.8,
            'claimed' => $stockWallet->claimed ?? 0,
            'unclaimed' => $stockWallet->unclaimed ?? 0,
        ]);
    }

    public function executeTrade(Request $request)
    {
        $request->validate([
            'stock_id' => 'required|exists:stocks,id',
            'trade_action' => ['required', Rule::in(['buy', 'sell'])],
            'order_type' => ['required', Rule::in(['market', 'limit', 'stop'])],
            'time_in_force' => ['required', Rule::in(['day', 'gtc'])],
            'shares_quantity' => 'required|numeric|min:0.01',
            'fiat_amount' => 'required|numeric|min:0.01',
            'limit_price' => 'nullable|numeric|min:0.01',
            'stop_price' => 'nullable|numeric|min:0.01',
            'fill_outside_rth' => 'boolean',
        ]);

        $user = Auth::user();
        $stock = Stock::findOrFail($request->stock_id);
        $tradeAction = $request->trade_action;
        $orderType = $request->order_type;

        // Determine account type based on stock market
        $accountType = $stock->market === 'gbx' ? 'gbx_balance' : 'usd_balance';
        $transactionAccount = $stock->market;

        DB::beginTransaction();

        try {
            // Determine execution price based on order type
            $executionPrice = $this->getExecutionPrice($stock, $orderType, $request);

            if (!$executionPrice) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to determine execution price'
                ], 400);
            }

            $stockTrade = StockTrade::create([
                'user_id' => $user->id,
                'stock_id' => $stock->id,
                'trade_action' => $tradeAction,
                'account_type' => $accountType,
                'order_type' => $orderType,
                'executed_at' => now(),
                'quantity' => 0, // Will be updated below
                'price_per_share' => $executionPrice,
            ]);

            if ($tradeAction === 'buy') {
                $result = $this->executeBuyOrder($user, $stock, $request, $executionPrice, $accountType, $transactionAccount, $stockTrade);
            } else {
                $result = $this->executeSellOrder($user, $stock, $request, $executionPrice, $accountType, $transactionAccount, $stockTrade);
            }

            if (!$result['success']) {
                DB::rollBack();
                return response()->json($result, 400);
            }

            DB::commit();
            return response()->json($result);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error processing trade: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing your trade. Please try again.'
            ], 500);
        }
    }

    private function getExecutionPrice(Stock $stock, string $orderType, Request $request): ?float
    {
        // switch ($orderType) {
            // case 'limit':
            //     return $request->limit_price;
            // case 'stop':
            //     return $request->stop_price;
            // case 'market':
            // default:
                // For market orders, use current stock price
                // In a real system, you'd use bid/ask prices
                return $stock->value;
        // }
    }

    private function executeBuyOrder($user, $stock, $request, $executionPrice, $accountType, $transactionAccount, $stockTrade)
    {
        $sharesQuantity = floatval($request->shares_quantity);
        $totalCost = $sharesQuantity * $executionPrice;

        // Check if user has sufficient funds
        $currentBalance = $user->wallet->balance ?? 0;
        if ($currentBalance < $totalCost) {
            return [
                'success' => false,
                'message' => 'Insufficient funds to complete this purchase.'
            ];
        }

        // Execute the buy order
        // $user->wallet()->decrement($accountType, $totalCost);
        $sourceWalletCoin = $user->wallet->walletCoins()->where('coin_id', 35)->first();
        $sourceWalletCoin->balance -= $totalCost;
        // dd($sourceWalletCoin->balance, $investedAmountInCoin, $request->invested_amount, $sourceWalletCoin->coin->value);
        $sourceWalletCoin->save();
        $user->stockWallets()->firstOrCreate(
            ['stock_id' => $stock->id],
            ['claimed' => 0, 'claimed' => 0]
        )->increment('claimed', $sharesQuantity);

        // Create transaction record
        Transaction::create([
            'user_id' => $user->id,
            'transaction_id' => $this->generateUniqueTransactionId(),
            'amount' => $totalCost,
            'type' => 'stock_buy',
            'details' => "Bought {$sharesQuantity} shares of {$stock->name} ({$stock->code}) at {$executionPrice}",
            'status_id' => 1,
            'action_at' => now(),
            'action_by' => $user->id,
            // 'account' => $transactionAccount,
            'stock_trade_id' => $stockTrade->id,
        ]);

        // Update stock trade record
        $stockTrade->update([
            'quantity' => $sharesQuantity,
            'price_per_share' => $executionPrice,
        ]);

        $currencySymbol = strtoupper($transactionAccount) === 'USD' ? '$' : '£';

        return [
            'success' => true,
            'message' => "Successfully bought {$sharesQuantity} shares of {$stock->name} ({$stock->code}) at {$currencySymbol}" . number_format($executionPrice * $sharesQuantity, 2),
            'data' => [
                'shares_purchased' => $sharesQuantity,
                'price_per_share' => $executionPrice,
                'total_cost' => $totalCost,
                'remaining_balance' => $currentBalance - $totalCost
            ]
        ];
    }

    private function executeSellOrder($user, $stock, $request, $executionPrice, $accountType, $transactionAccount, $stockTrade)
    {
        $sharesQuantity = floatval($request->shares_quantity);
        $userStockWallet = $user->stockWallets()->where('stock_id', $stock->id)->first();

        // Check if user has sufficient shares
        if (!$userStockWallet || $userStockWallet->claimed < $sharesQuantity) {
            return [
                'success' => false,
                'message' => 'Insufficient shares to complete this sale.'
            ];
        }

        $totalRevenue = $sharesQuantity * $executionPrice;

        // Execute the sell order
        $userStockWallet->decrement('claimed', $sharesQuantity);

        // $user->wallet()->increment($accountType, $totalRevenue);
        $sourceWalletCoin = $user->wallet->walletCoins()->where('coin_id', 35)->first();
        $sourceWalletCoin->balance += $totalRevenue;
        // dd($sourceWalletCoin->balance, $investedAmountInCoin, $request->invested_amount, $sourceWalletCoin->coin->value);
        $sourceWalletCoin->save();

        $userStockWallet->increment('sold', $sharesQuantity);

        // Create transaction record
        Transaction::create([
            'user_id' => $user->id,
            'transaction_id' => $this->generateUniqueTransactionId(),
            'amount' => $totalRevenue,
            'type' => 'stock_sell',
            'details' => "Sold {$sharesQuantity} shares of {$stock->name} ({$stock->code}) at {$executionPrice}",
            'status_id' => 1,
            'action_at' => now(),
            'action_by' => $user->id,
            'account' => $transactionAccount,
            'stock_trade_id' => $stockTrade->id,
        ]);

        // Update stock trade record
        $stockTrade->update([
            'quantity' => $sharesQuantity,
            'price_per_share' => $executionPrice,
        ]);

        $currencySymbol = strtoupper($transactionAccount) === 'USD' ? '$' : '£';

        return [
            'success' => true,
            'message' => "Successfully sold {$sharesQuantity} shares of {$stock->name} ({$stock->code}) at {$currencySymbol}" . number_format($executionPrice * $sharesQuantity, 2),
            'data' => [
                'shares_sold' => $sharesQuantity,
                'price_per_share' => $executionPrice,
                'total_revenue' => $totalRevenue,
                'remaining_shares' => $userStockWallet->claimed - $sharesQuantity
            ]
        ];
    }
}
