<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\StockTrade;
use App\Models\ForexTrade;
use App\Models\ForexWallet;
use App\Models\ForexPair;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TradeHistoryController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        // 1. Fetch Stock History
        $stockTrades = StockTrade::where('user_id', $user->id)
                        ->with('stock')
                        ->orderBy('created_at', 'desc')
                        ->get();

        // 2. Fetch Forex History
        $forexTrades = ForexTrade::where('user_id', $user->id)
                        ->with('pair')
                        ->orderBy('created_at', 'desc')
                        ->get();

        // 3. Fetch Open Forex Positions (Net Holdings > 0)
        $openForexPositions = ForexWallet::where('user_id', $user->id)
                                ->where('balance', '>', 0)
                                ->with('pair')
                                ->get();

        return view('trades.history', compact('stockTrades', 'forexTrades', 'openForexPositions'));
    }

    public function closeForexPosition(Request $request)
    {
        $request->validate([
            'wallet_id' => 'required|exists:forex_wallets,id',
        ]);

        $user = Auth::user();
        $wallet = ForexWallet::where('user_id', $user->id)
                    ->where('id', $request->wallet_id)
                    ->firstOrFail();

        if ($wallet->balance <= 0) {
            return back()->with('error', 'No open position to close.');
        }

        // Logic to "Close" position: Sell entire balance at current rate
        $pair = $wallet->pair;
        $currentPrice = $pair->exchange_rate;
        $units = $wallet->balance;

        // Calculate return value based on simple spot trading logic
        // (If using margin, math changes to: Margin + PnL)
        $returnValue = $units * $currentPrice;
        $pnl = ($currentPrice - $wallet->avg_entry_price) * $units;

        DB::beginTransaction();
        try {
            // Credit User Wallet
            $user->wallet->increment('balance', $returnValue);

            // Record "Sell" Trade
            ForexTrade::create([
                'user_id' => $user->id,
                'forex_pair_id' => $pair->id,
                'type' => 'sell',
                'entry_price' => $currentPrice,
                'quantity' => $units,
                'leverage_used' => $wallet->avg_leverage,
                'margin_amount' => 0,
            ]);

            // Reset Wallet Position
            $wallet->update([
                'balance' => 0,
                'avg_entry_price' => 0,
                'avg_leverage' => 1
            ]);

            DB::commit();
            return back()->with('success', "Position for {$pair->symbol} closed. P/L: $" . number_format($pnl, 2));

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to close position: ' . $e->getMessage());
        }
    }
}
