<?php

namespace App\Http\Controllers;

use App\Models\Signal;
use App\Models\UserSignalSubscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SignalController extends Controller
{
    /**
     * Display a listing of available signals for users to subscribe.
     */
    public function index()
    {
        $signals = Signal::where('status', 'active')->get();

        // Pass any success/error messages from session
        $success = session('success');
        $error = session('error');

        return view('signals.index', compact('signals', 'success', 'error'));
    }

    /**
     * Handle user subscription to a signal.
     */
    public function subscribe(Request $request)
    {
        $request->validate([
            'signal_id' => 'required|exists:signals,id',
            'duration_value' => 'required|integer|min:1', // Validation ensures it's an integer
            'duration_unit' => ['required', 'string', 'in:day,week,month,year'],
        ]);

        $signal = Signal::find($request->signal_id);
        $user = Auth::user();

        // Explicitly cast duration_value to integer before passing to Carbon
        $durationValue = (int) $request->duration_value;

        $durationUnit = $request->duration_unit;

        // Price calculation: price_per_unit is the price for ONE default_duration_value unit.
        // If user selects `durationValue` units, it's `price_per_unit * durationValue`.
        $paidPrice = $signal->price_per_unit * $durationValue;

        // Check if user has sufficient balance in their USD wallet
        $usdWalletCoin = $user->wallet->walletCoins()->whereHas('coin', function($query) {
            $query->where('code', 'USD');
        })->first();

        if (!$usdWalletCoin || $usdWalletCoin->balance < $paidPrice) {
            return redirect()->back()->with('error', 'Insufficient USD balance to subscribe to this signal.');
        }

        DB::beginTransaction();
        try {
            // Deduct funds from user's USD wallet
            $usdWalletCoin->balance -= $paidPrice;
            $usdWalletCoin->save();

            // Calculate start and end dates
            $startDate = Carbon::now();
            $endDate = $startDate->copy()->add($durationValue, $durationUnit); // Carbon handles different units

            // Create subscription record
            UserSignalSubscription::create([
                'user_id' => $user->id,
                'signal_id' => $signal->id,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'paid_price' => $paidPrice,
                'status' => 'active',
            ]);

            DB::commit();
            return redirect()->route('signals.index')->with('success', 'Successfully subscribed to ' . $signal->name . ' for $' . number_format($paidPrice, 2) . '!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to subscribe to signal. Please try again. ' . $e->getMessage());
        }
    }

    /**
     * Display a listing of user's active/past signal subscriptions.
     */
    public function mySignals()
    {
        $user = Auth::user();
        $subscriptions = $user->userSignalSubscriptions()->with('signal')->latest()->get();

        return view('signals.my-signals', compact('subscriptions'));
    }
}
