<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProfileUpdateRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;
use App\Models\KycStatus;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class ProfileController extends Controller
{
    /**
     * Display the user's profile form.
     */
    public function edit(Request $request): View
    {
        return view('profile.edit', [
            'user' => $request->user(),
        ]);
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): RedirectResponse
    {
        $request->user()->fill($request->validated());

        if ($request->user()->isDirty('email')) {
            $request->user()->email_verified_at = null;
        }

        $request->user()->save();

        return Redirect::route('profile.edit')->with('status', 'profile-updated');
    }

    public function kycUpdate(Request $request): RedirectResponse
    {
        $validator = Validator::make($request->all(), [
            'kyc_picture' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();

        try {
            $user = Auth::user();

            if ($request->hasFile('kyc_picture')) {
                // Delete old KYC picture if exists
                if ($user->kyc_picture) {
                    Storage::disk('public')->delete($user->kyc_picture);
                }

                // Store the new file using the same name as before
                $fileName = time() . '_' . $request->file('kyc_picture')->getClientOriginalName();
                $request->file('kyc_picture')->storeAs('uploads', $fileName);

                $user->kyc = $fileName;

                // Update KYC status to pending
                $pendingStatus = KycStatus::where('name', 'pending')->firstOrFail();
                $user->kyc_status_id = $pendingStatus->id;

                $user->save();
            }

            DB::commit();

            return redirect()->back()->with('status', 'kyc-updated');

        } catch (\Exception $e) {
            DB::rollBack();
            // Handle the error (e.g., log it, return an error message)
            throw $e;
        }
    }
    /**
     * Delete the user's account.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return Redirect::to('/');
    }
}
