<?php

namespace App\Http\Controllers;

use App\Models\ExpertTrader;
use App\Models\UserReplication;
use App\Models\User; // To access user's wallet
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class CopyTradingController extends Controller
{
    /**
     * Display a listing of available expert traders for users to replicate.
     */
    public function index()
    {
        $expertTraders = ExpertTrader::where('status', 'active')->get();

        // Pass any session messages
        $success = session('success');
        $error = session('error');

        return view('copytrading.index', compact('expertTraders', 'success', 'error'));
    }

    /**
     * Handle user starting replication of an expert trader.
     */
    public function startReplication(Request $request)
    {
        $request->validate([
            'expert_trader_id' => 'required|exists:expert_traders,id',
            'allocated_amount' => 'required|numeric|min:1', // Minimum allocation amount
            // You might add validation for risk settings here
        ]);

        $expertTrader = ExpertTrader::find($request->expert_trader_id);
        $user = Auth::user();

        // Check if user already replicating this expert
        if ($user->userReplications()->where('expert_trader_id', $expertTrader->id)->where('status', 'active')->exists()) {
            return redirect()->back()->with('error', 'You are already replicating this expert trader.');
        }

        // Check if user has sufficient balance in their USD wallet
        $usdWalletCoin = $user->wallet->walletCoins()->whereHas('coin', function($query) {
            $query->where('code', 'USD');
        })->first();

        if (!$usdWalletCoin || $usdWalletCoin->balance < $request->allocated_amount) {
            return redirect()->back()->with('error', 'Insufficient USD balance to allocate this amount.');
        }

        DB::beginTransaction();
        try {
            // Deduct allocated amount from user's USD wallet (or mark it as 'in use')
            // For true copy trading, this amount might not leave the wallet directly but be 'reserved'
            // For simplicity, we'll reduce the balance for now, assuming it's transferred to a copy-trade specific balance
            $usdWalletCoin->balance -= $request->allocated_amount;
            $usdWalletCoin->save();

            UserReplication::create([
                'user_id' => $user->id,
                'expert_trader_id' => $expertTrader->id,
                'allocated_amount' => $request->allocated_amount,
                'current_pnl' => 0, // Starts at 0 P&L
                'status' => 'active',
            ]);

            DB::commit();
            return redirect()->route('copytrading.index')->with('success', 'Successfully started replicating ' . $expertTrader->name . ' with $' . number_format($request->allocated_amount, 2) . '!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to start replication. Please try again. ' . $e->getMessage());
        }
    }

    /**
     * Handle user stopping replication of an expert trader.
     */
    public function stopReplication(Request $request, UserReplication $userReplication)
    {
        // Ensure the user owns this replication
        if ($userReplication->user_id !== Auth::id()) {
            return redirect()->back()->with('error', 'You are not authorized to stop this replication.');
        }

        if ($userReplication->status !== 'active' && $userReplication->status !== 'paused') {
            return redirect()->back()->with('error', 'Replication is not in a state that can be stopped.');
        }

        DB::beginTransaction();
        try {
            // Here you would typically transfer the 'allocated_amount' + 'current_pnl' back to the user's wallet
            // For simplicity, we'll just update status. The actual fund transfer logic needs to be robust.
            $user->wallet->walletCoins()->whereHas('coin', function($query) {
                $query->where('code', 'USD');
            })->first()->increment('balance', $userReplication->allocated_amount + $userReplication->current_pnl);


            $userReplication->update([
                'status' => 'stopped',
                'current_pnl' => $userReplication->current_pnl, // Final P&L at stop
                // Additional fields like `stopped_at` could be added to migration
            ]);

            DB::commit();
            return redirect()->route('copytrading.my-copytrades')->with('success', 'Successfully stopped replicating ' . $userReplication->expertTrader->name . '.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to stop replication. Please try again. ' . $e->getMessage());
        }
    }


    /**
     * Display a listing of user's active/past replications.
     */
    public function myCopyTrades()
    {
        $user = Auth::user();
        $replications = $user->userReplications()->with('expertTrader')->latest()->get();

        return view('copytrading.my-copytrades', compact('replications'));
    }
}
