<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Signal;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class AdminSignalController extends Controller
{
    /**
     * Display a listing of the signals.
     */
    public function index()
    {
        $signals = Signal::latest()->get(); // Fetch all signals
        return view('admin-dash.signals.index', compact('signals'));
    }

    /**
     * Store a newly created signal in storage.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:signals,name',
            'description' => 'nullable|string',
            'price_per_unit' => 'required|numeric|min:0.01',
            'duration_unit' => ['required', 'string', Rule::in(['day', 'week', 'month', 'year'])],
            'default_duration_value' => 'required|integer|min:1',
            'roi_percentage' => 'required|numeric|min:0|max:100',
            'max_subscribers' => 'required|integer|min:0',
            'status' => ['required', 'string', Rule::in(['active', 'inactive'])],
        ]);

        Signal::create($validatedData);

        return redirect()->route('admin.signals.index')->with('success', 'Signal added successfully!');
    }

    /**
     * Update the specified signal in storage.
     */
    public function update(Request $request, Signal $signal)
    {
        $validatedData = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('signals')->ignore($signal->id)],
            'description' => 'nullable|string',
            'price_per_unit' => 'required|numeric|min:0.01',
            'duration_unit' => ['required', 'string', Rule::in(['day', 'week', 'month', 'year'])],
            'default_duration_value' => 'required|integer|min:1',
            'roi_percentage' => 'required|numeric|min:0|max:100',
            'max_subscribers' => 'required|integer|min:0',
            'status' => ['required', 'string', Rule::in(['active', 'inactive'])],
        ]);

        $signal->update($validatedData);

        return redirect()->route('admin.signals.index')->with('success', 'Signal updated successfully!');
    }

    /**
     * Remove the specified signal from storage.
     */
    public function destroy(Signal $signal)
    {
        $signal->delete(); // Soft delete
        return redirect()->route('admin.signals.index')->with('success', 'Signal deleted successfully!');
    }
}
