<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\InvestmentPlan;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class AdminInvestmentPlanController extends Controller
{
    /**
     * Display a listing of the investment plans.
     */
    public function index()
    {
        $investmentPlans = InvestmentPlan::latest()->get();
        return view('admin.investment-plans.index', compact('investmentPlans'));
    }

    /**
     * Store a newly created investment plan.
     */
    public function store(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'name' => 'required|string|max:255|unique:investment_plans,name',
                'description' => 'nullable|string',
                'min_investment' => 'required|numeric|min:0',
                'max_investment' => 'required|numeric|min:' . $request->input('min_investment'),
                'daily_roi_percentage' => 'required|numeric|min:0|max:100',
                'min_duration_value' => 'required|integer|min:1',
                'max_duration_value' => 'required|integer|min:' . $request->input('min_duration_value'),
                'allowed_duration_units' => 'required|array',
                'allowed_duration_units.*' => ['required', 'string', Rule::in(['days', 'weeks', 'months', 'years'])],
                'features' => 'nullable|array',
                'features.*' => 'nullable|string|max:255',
            ]);

            // Calculate total_roi_percentage (e.g., based on max duration to show full potential)
            $totalDurationForMaxRoi = 0;
            switch($validatedData['allowed_duration_units'][0] ?? 'days') {
                case 'days': $totalDurationForMaxRoi = $validatedData['max_duration_value']; break;
                case 'weeks': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 7; break;
                case 'months': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 30; break; /* Approx */
                case 'years': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 365; /* Approx */ break;
            }

            $validatedData['total_roi_percentage'] = $validatedData['daily_roi_percentage'] * $totalDurationForMaxRoi;

            $validatedData['features'] = array_values(array_filter($validatedData['features'] ?? []));

            InvestmentPlan::create($validatedData);

            return redirect()->route('admin.investment-plans.index')->with('success', 'Investment Plan added successfully!');
        } catch (ValidationException $e) {
            // Collect all validation error messages into a single array
            $errors = $e->validator->errors()->all();
            // Redirect back with errors flashed to the session
            return redirect()->back()->withErrors($errors)->withInput();
        }
    }

    /**
     * Update the specified investment plan.
     */
    public function update(Request $request, InvestmentPlan $investmentPlan)
    {
        try {
            $validatedData = $request->validate([
                'name' => ['required', 'string', 'max:255', Rule::unique('investment_plans')->ignore($investmentPlan->id)],
                'description' => 'nullable|string',
                'min_investment' => 'required|numeric|min:0',
                'max_investment' => 'required|numeric|min:' . $request->input('min_investment'),
                'daily_roi_percentage' => 'required|numeric|min:0|max:100',
                'min_duration_value' => 'required|integer|min:1',
                'max_duration_value' => 'required|integer|min:' . $request->input('min_duration_value'),
                'allowed_duration_units' => 'required|array',
                'allowed_duration_units.*' => ['required', 'string', Rule::in(['days', 'weeks', 'months', 'years'])],
                'features' => 'nullable|array',
                'features.*' => 'nullable|string|max:255',
            ]);

            // Recalculate total_roi_percentage based on max duration
            $totalDurationForMaxRoi = 0;
            switch($validatedData['allowed_duration_units'][0] ?? 'days') {
                case 'days': $totalDurationForMaxRoi = $validatedData['max_duration_value']; break;
                case 'weeks': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 7; break;
                case 'months': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 30; break;
                case 'years': $totalDurationForMaxRoi = $validatedData['max_duration_value'] * 365; break;
            }
            $validatedData['total_roi_percentage'] = $validatedData['daily_roi_percentage'] * $totalDurationForMaxRoi;

            $validatedData['features'] = array_values(array_filter($validatedData['features'] ?? []));

            $investmentPlan->update($validatedData);

            return redirect()->route('admin.investment-plans.index')->with('success', 'Investment Plan updated successfully!');
        } catch (ValidationException $e) {
            $errors = $e->validator->errors()->all();
            return redirect()->back()->withErrors($errors)->withInput();
        }
    }

    /**
     * Remove the specified investment plan from storage (soft delete).
     */
    public function destroy(InvestmentPlan $investmentPlan)
    {
        // Consider logic here: What happens to active user_fixed_investments associated with this plan?
        // You might want to prevent deletion if there are active investments.
        $investmentPlan->delete();
        return redirect()->route('admin.investment-plans.index')->with('success', 'Investment Plan deleted successfully!');
    }
}
