<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ExpertTrader;
use App\Models\UserReplication;
// Removed use App\Models\User; // No longer linking to User model directly
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;

class AdminExpertTraderController extends Controller
{
    /**
     * Display a listing of expert traders.
     */
    public function index()
    {
        $expertTraders = ExpertTrader::latest()->get();
        // Removed fetching nonExpertUsers as experts are now independent.

        return view('admin-dash.expert-traders.index', compact('expertTraders'));
    }

    /**
     * Store a newly created expert trader.
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255|unique:expert_traders,name',
            'email' => 'required|email|max:255|unique:expert_traders,email',

            // New Investment/Duration/Risk fields
            'min_investment_amount' => 'required|numeric|min:0',
            // Max investment must be greater than or equal to min investment
            'max_investment_amount' => 'required|numeric|min:0|gte:min_investment_amount',
            'expected_duration_value' => 'required|integer|min:1',
            'expected_duration_unit' => ['required', 'string', Rule::in(['days', 'weeks', 'months', 'years'])],
            'risk_level' => ['required', 'string', Rule::in(['Low', 'Medium', 'High'])],

            // Features (nullable array, max 3 items)
            'features' => 'nullable|array|max:3',
            'features.*' => 'nullable|string|max:255',

            // ROI and Share fields
            'expected_roi_percentage' => 'required|numeric|min:0|max:100',
            'revenue_share_percentage' => 'required|numeric|min:0|max:100',

            // Description and Status
            'description' => 'nullable|string',
            'status' => ['required', 'string', Rule::in(['active', 'suspended'])],
        ]);

        // Clean up features array: remove nulls/empty strings
        if (isset($validatedData['features'])) {
            $validatedData['features'] = array_values(array_filter($validatedData['features']));
        }

        ExpertTrader::create($validatedData);

        return redirect()->route('admin.expert-traders.index')->with('success', 'Copy-Trade Plan added successfully!');
    }

    /**
     * Update the specified expert trader plan.
     */
    public function update(Request $request, ExpertTrader $expertTrader)
    {
        $validatedData = $request->validate([
            // Unique check must ignore the current expert trader's ID
            'name' => ['required', 'string', 'max:255', Rule::unique('expert_traders', 'name')->ignore($expertTrader->id)],
            'email' => ['required', 'email', 'max:255', Rule::unique('expert_traders', 'email')->ignore($expertTrader->id)],

            // New Investment/Duration/Risk fields
            'min_investment_amount' => 'required|numeric|min:0',
            'max_investment_amount' => 'required|numeric|min:0|gte:min_investment_amount',
            'expected_duration_value' => 'required|integer|min:1',
            'expected_duration_unit' => ['required', 'string', Rule::in(['days', 'weeks', 'months', 'years'])],
            'risk_level' => ['required', 'string', Rule::in(['Low', 'Medium', 'High'])],

            // Features (nullable array, max 3 items)
            'features' => 'nullable|array|max:3',
            'features.*' => 'nullable|string|max:255',

            // ROI and Share fields
            'expected_roi_percentage' => 'required|numeric|min:0|max:100',
            'revenue_share_percentage' => 'required|numeric|min:0|max:100',

            // Description and Status
            'description' => 'nullable|string',
            'status' => ['required', 'string', Rule::in(['active', 'suspended'])],
        ]);

        // Clean up features array: remove nulls/empty strings
        if (isset($validatedData['features'])) {
            $validatedData['features'] = array_values(array_filter($validatedData['features']));
        }

        $expertTrader->update($validatedData);

        return redirect()->route('admin.expert-traders.index')->with('success', 'Copy-Trade Plan updated successfully!');
    }

    /**
     * Remove the specified expert trader from storage (soft delete).
     */
    public function destroy(ExpertTrader $expertTrader)
    {
        // Consider logic here: What happens to active replications if expert is deleted?
        // You might want to 'stop' all associated user_replications first.
        $expertTrader->delete();
        return redirect()->route('admin.expert-traders.index')->with('success', 'Expert Trader deleted successfully!');
    }

    public function subscriptions(ExpertTrader $expertTrader = null)
    {
        // Start query with necessary relationships
        $query = UserReplication::with(['user', 'expertTrader']);

        // If an expert is passed (via the link in the table), filter by that expert
        if ($expertTrader) {
            $query->where('expert_trader_id', $expertTrader->id);
        }

        // Paginate results
        $replications = $query->latest()->paginate(20);

        return view('admin-dash.expert-traders.subscriptions', compact('replications', 'expertTrader'));
    }
}
