<?php

namespace App\Console\Commands;

use App\Models\Stock;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class UpdateStockData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stocks:update-av-daily
                            {--symbol= : Update a specific stock symbol}
                            {--force-all : Force update all stocks}
                            {--specific : Update only specific stocks}'; // Added --specific option

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update stock data daily using Alpha Vantage API, updating specific IDs and the 3 oldest if not recently updated, handling .L suffix.  Can also update only specific stocks.';

    /**
     * The Alpha Vantage API key.
     *
     * @var string
     */
    protected $apiKey;

    /**
     * Market symbol mappings for Alpha Vantage.
     * Adjust as needed based on Alpha Vantage's requirements.
     *
     * @var array
     */
    protected $exchangeMappings = [
        'gbx' => 'LSE', // London Stock Exchange
        'usd' => null,   // Default US exchange (no suffix needed for primary listing)
    ];

    /**
     * The spread percentage to apply for bid and ask values.
     * Adjust this as needed.
     *
     * @var float
     */
    protected $spreadPercentage = 0.001; // 0.1% spread

    /**
     * The specific stock IDs to always attempt to update.
     *
     * @var array
     */
    protected $specificStockIds = [19, 21, 41];

    /**
     * The number of oldest stocks to attempt to update.
     *
     * @var int
     */
    protected $numberOfOldestToUpdate = 1; // !temp

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->apiKey = config('services.alphavantage.api_key');

        if (empty($this->apiKey)) {
            $this->error('Alpha Vantage API key not found. Please add it to your .env file as ALPHAVANTAGE_API_KEY');
            return Command::FAILURE;
        }

        $symbol = $this->option('symbol');
        $forceAll = $this->option('force-all');
        $specificOnly = $this->option('specific'); // Get the new option

        if ($symbol) {
            $stock = Stock::where('code', $symbol)->first();
            if (!$stock) {
                $this->error("Stock with symbol {$symbol} not found");
                return Command::FAILURE;
            }
            $this->updateStock($stock);
        } else {
            if ($specificOnly) {
                $this->updateSpecificStocks(); // Call new method
            } else {
                $this->updateSelectedAndOldestStocks($forceAll);
            }
        }

        return Command::SUCCESS;
    }

    /**
     * Update a single stock record and log changes using Alpha Vantage.
     *
     * @param \App\Models\Stock $stock
     * @return void
     */
    protected function updateStock(Stock $stock)
    {
        $this->info("Updating stock: {$stock->name} ({$stock->code}) - Market: {$stock->market} (ID: {$stock->id})");

        try {
            // Store original values for comparison
            $originalValues = $stock->only(['value', 'ask_value', 'bid_value', 'year_range_min', 'year_range_max']);
            $updatedValues = [];

            // Format symbol for the correct exchange for Alpha Vantage
            $formattedSymbol = $this->formatSymbolForExchange($stock->code, $stock->market);

            // Get quote data from Alpha Vantage
            $quoteData = $this->fetchQuoteData($formattedSymbol);

            if (!$quoteData || !isset($quoteData['Global Quote']) || isset($quoteData['Information'])) {
                $this->warn("Unable to fetch quote data for {$formattedSymbol} from Alpha Vantage: " . ($quoteData['Information'] ?? 'No detailed error'));
                return;
            }
            $globalQuote = $quoteData['Global Quote'];

            $currentValue = $globalQuote['05. price'] ?? null;
            $newAskValue = null;
            $newBidValue = null;

            if ($currentValue !== null) {
                $spreadAmount = $currentValue * $this->spreadPercentage;
                $newAskValue = round($currentValue + $spreadAmount, 2);
                $newBidValue = round($currentValue - $spreadAmount, 2);
            }

            // Update stock model and track changes
            if ($currentValue !== null && $stock->value != $currentValue) {
                $updatedValues['value'] = $currentValue;
                $stock->value = $currentValue;
            }

            if ($newAskValue !== null && $stock->ask_value != $newAskValue) {
                $updatedValues['ask_value'] = $newAskValue;
                $stock->ask_value = $newAskValue;
            }

            if ($newBidValue !== null && $stock->bid_value != $newBidValue) {
                $updatedValues['bid_value'] = $newBidValue;
                $stock->bid_value = $newBidValue;
            }

            // Get 52-week range data
            $rangeData = $this->fetchRangeData($formattedSymbol);

            if ($rangeData) {
                if (isset($rangeData['52WeekLow']) && $stock->year_range_min != $rangeData['52WeekLow']) {
                    $updatedValues['year_range_min'] = $rangeData['52WeekLow'];
                    $stock->year_range_min = $rangeData['52WeekLow'];
                }
                if (isset($rangeData['52WeekHigh']) && $stock->year_range_max != $rangeData['52WeekHigh']) {
                    $updatedValues['year_range_max'] = $rangeData['52WeekHigh'];
                    $stock->year_range_max = $rangeData['52WeekHigh'];
                }
            }

            if (!empty($updatedValues)) {
                $stock->updated_at = now();
                $stock->save();
                $logMessage = "Updated {$stock->code} (Alpha Vantage): ";
                foreach ($updatedValues as $key => $newValue) {
                    $logMessage .= "{$key} from {$originalValues[$key]} to {$newValue}, ";
                }
                $logMessage .= "at " . now();
                $this->info($logMessage);
                Log::info($logMessage);
            } else {
                $this->info("No changes for {$stock->code} (Alpha Vantage) at " . now());
                Log::info("No changes for {$stock->code} (Alpha Vantage) at " . now());
            }
        } catch (\Exception $e) {
            Log::error("Error updating stock {$stock->code} (Alpha Vantage): " . $e->getMessage());
            $this->error("Error updating {$stock->code} (Alpha Vantage): " . $e->getMessage());
        }
    }

    /**
     * Format symbol for the correct exchange for Alpha Vantage, handling existing .L.
     *
     * @param string $symbol
     * @param string $market
     * @return string
     */
    protected function formatSymbolForExchange($symbol, $market)
    {
        // Use regex to remove .L suffix if it exists
        $cleanSymbol = strtoupper(preg_replace('/\.L$/', '', $symbol));

        if (!$market || !isset($this->exchangeMappings[$market])) {
            return $cleanSymbol;
        }

        $exchange = $this->exchangeMappings[$market];

        // Format for London Stock Exchange for Alpha Vantage
        if ($exchange === 'LSE') {
            return "{$cleanSymbol}.L";
        }

        return $cleanSymbol;
    }

    /**
     * Update the specific stock IDs and the oldest stocks.
     *
     * @param bool $forceAll
     * @return void
     */
    protected function updateSelectedAndOldestStocks(bool $forceAll = false)
    {
        $stocksToUpdate = Stock::whereIn('id', $this->specificStockIds)
            ->get()
            ->filter(function ($stock) use ($forceAll) {
                return $forceAll || !$stock->updated_at || $stock->updated_at->diffInHours(now()) >= 12;
            });

        $numberOfSpecificToUpdate = $stocksToUpdate->count();
        $remainingToUpdate = max(0, 4 - $numberOfSpecificToUpdate);

        if ($remainingToUpdate > 0) {
            $oldestStocks = Stock::whereNotIn('id', $this->specificStockIds)
                ->orderBy('updated_at', 'asc')
                ->take($remainingToUpdate)
                ->get()
                ->filter(function ($stock) use ($forceAll) {
                    return $forceAll || !$stock->updated_at || $stock->updated_at->diffInHours(now()) >= 24;
                });
            $stocksToUpdate = $stocksToUpdate->concat($oldestStocks)->unique('id')->values();
        }

        $this->info("Attempting to update {$stocksToUpdate->count()} stocks using Alpha Vantage");

        $progress = $this->output->createProgressBar($stocksToUpdate->count());
        $progress->start();

        $apiCallCount = 0;
        foreach ($stocksToUpdate as $stock) {
            $this->updateStock($stock);
            $apiCallCount++;

            // Respect Alpha Vantage's rate limit (25 requests per day on free tier)
            if ($apiCallCount % 5 == 0) {
                sleep(60); // Sleep for 60 seconds (1 minute)
            }

            $progress->advance();
        }

        $progress->finish();
        $this->newLine();
        $this->info('Selected and oldest stock update process completed using Alpha Vantage');
    }

    /**
     * Update only the specific stock IDs.
     *
     * @return void
     */
    protected function updateSpecificStocks()
    {
        $stocksToUpdate = Stock::whereIn('id', $this->specificStockIds)->get();

        $this->info("Attempting to update " . count($this->specificStockIds) . " specific stocks using Alpha Vantage");

        $progress = $this->output->createProgressBar($stocksToUpdate->count());
        $progress->start();

        $apiCallCount = 0;
        foreach ($stocksToUpdate as $stock) {
            $this->updateStock($stock);
            $apiCallCount++;

            // Respect Alpha Vantage's rate limit.
            if ($apiCallCount % 5 == 0) {
                sleep(60);
            }
            $progress->advance();
        }

        $progress->finish();
        $this->newLine();
        $this->info('Specific stocks update process completed.');
    }

    /**
     * Fetch quote data from Alpha Vantage API.
     *
     * @param string $symbol
     * @return array|null
     */
    protected function fetchQuoteData($symbol)
    {
        $response = Http::get('https://www.alphavantage.co/query', [
            'function' => 'GLOBAL_QUOTE',
            'symbol' => $symbol,
            'apikey' => $this->apiKey,
        ]);

        $this->info("Query link: https://www.alphavantage.co/query?function=GLOBAL_QUOTE&symbol={$symbol}apikey={$this->apiKey}" . now()); // !temp

        if ($response->successful()) {
            $data = $response->json();
            Log::info("Alpha Vantage Quote data for {$symbol}: " . json_encode($data));
            return $data;
        }

        Log::error("Alpha Vantage API error (quote): " . $response->body());
        return null;
    }

    /**
     * Fetch 52-week range data from Alpha Vantage API.
     *
     * @param string $symbol
     * @return array|null
     */
    protected function fetchRangeData($symbol)
    {
        $response = Http::get('https://www.alphavantage.co/query', [
            'function' => 'GLOBAL_QUOTE',
            'symbol' => $symbol,
            'apikey' => $this->apiKey,
        ]);

        if ($response->successful()) {
            $data = $response->json();
            if (isset($data['Global Quote'])) {
                return [
                    '52WeekLow' => $data['Global Quote']['52 week low'] ?? null,
                    '52WeekHigh' => $data['Global Quote']['52 week high'] ?? null,
                ];
            }
        }

        Log::warning("Alpha Vantage unable to fetch 52-week range for {$symbol}: " . $response->body());
        return null;
    }
}
